/*
 * Toolkit GUI, an application built for operating pinkRF's signal generators.
 *
 * Contact: https://www.pinkrf.com/contact/
 * Copyright © 2018-2024 pinkRF B.V
 * GNU General Public License version 3.
 *
 * This program is free software: you can redistribute it and/or modify it under the terms of the GNU General Public License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later version.
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 * You should have received a copy of the GNU General Public License along with this program. If not, see https://www.gnu.org/licenses/
 *
 * Author: Iordan Svechtarov
 */

#include <math.h>
#include "miscellaneous.h"

/**********************************************************************************************************************************************************************************
 * Zero Chopper; Cleans up the extra zeroes at the end of a number (e.g. 150.1200 -> 150.12)
 **********************************************************************************************************************************************************************************/
QString zeroChopper(QString input)
{
	while(input.endsWith('0')){
		input.chop(1);
	}
	if(input.endsWith('.')){
		input.chop(1);
	}
	return input;
}

/**********************************************************************************************************************************************************************************
 * Unit Conversions
 **********************************************************************************************************************************************************************************/
double convert_dbm_to_watt(double value_in_dBm)
{
	double value_in_Watt = 0.001 * pow(10, 0.1 * value_in_dBm);
	return value_in_Watt;
}

double convert_watt_to_dbm(double value_in_watt)
{
	double value_in_dBm = (10 * log10(value_in_watt)) + 30;
	return value_in_dBm;
}

double convert_percent_to_dB(double percent)
{
	return convert_multiplier_to_dB(percent * 0.01);
}

double convert_dB_to_percent(double dB)
{
	return convert_dB_to_multiplier(dB) * 100;
}

double convert_multiplier_to_dB(double multiplier)
{
	double value_in_dB = 10 * log10(multiplier);
	return value_in_dB;
}

double convert_dB_to_multiplier(double dB)
{
	double value_in_multiplier = pow(10,(dB/10));
	return value_in_multiplier;
}

/**********************************************************************************************************************************************************************************
 * Version Compatibility Check
 **********************************************************************************************************************************************************************************/
bool isVersionCompatible(std::array<int,4> current, std::array<int,4> required)
{
	bool version_compatibility = true;
	for(unsigned int i = 0; i < current.size(); i++)
	{
		/* if something is clearly smaller or larger, break the loop, a conclusion can be made.
		 * if the numbers are partially similar (1.23.7 vs 1.23.9 or something like that) the loops can continue to the next cycle. */
		if(current.at(i) > required.at(i))
		{
			break;
		}
		else if (current.at(i) == required.at(i))
		{
			/* This isn't pointless. Prevents a premature break; Ensures the function keeps looping through every digit in version numbers if digits are identical. */
		}
		else
		{
			version_compatibility = false;
			break;
		}
	}

	return version_compatibility;
}
